package stanhebben.zenscript.parser;

import static java.util.Arrays.sort;

/**
 * Represents a compiled DFA. A compiled DFA has a compact representation and is
 * immediately usable for efficient processing.
 * <p>
 * A compiled DFA can be converted to a compact integer array. This array can
 * then be hardcoded in an application.
 *
 * @author Stan Hebben
 */
public class CompiledDFA {

    public static final int NOFINAL = Integer.MIN_VALUE;

    public HashMapII[] transitions;
    public int[] finals;

    /**
     * Constructs a compiled DFA from the specified transition graph and finals
     * arrays.
     * <p>
     * The transition array specifies all transitions for each state. The finals
     * array specifies the final class index of each state, or NOFINAL if the
     * state is not a final. There can multiple final classes, which can, for
     * example, be used to distinguish token types.
     *
     * @param transitions transitions graph
     * @param finals      finals
     */
    public CompiledDFA(HashMapII[] transitions, int[] finals) {
        this.transitions = transitions;
        this.finals = finals;
    }

    /**
     * Converts an integer array with a DFA definition (generated by toArray())
     * to a CompiledDFA.
     *
     * @param definition definition array
     */
    public CompiledDFA(int[] definition) {
        int ix = 0;
        int numStates = definition[ix++];
        transitions = new HashMapII[numStates];
        finals = new int[numStates];

        for(int i = 0; i < numStates; i++) {
            transitions[i] = new HashMapII();
            finals[i] = definition[ix++];

            int numRanges = definition[ix++];
            for(int j = 0; j < numRanges; j++) {
                int from = definition[ix++];
                int to = definition[ix++];
                int state = definition[ix++];
                for(int k = from; k <= to; k++) {
                    transitions[i].put(k, state);
                }
            }

            int numSingles = definition[ix++];
            for(int j = 0; j < numSingles; j++) {
                int label = definition[ix++];
                int state = definition[ix++];
                transitions[i].put(label, state);
            }
        }
    }

    /**
     * Converts the DFA to an integer array.
     *
     * @return an integer array representation of the DFA
     */
    public int[] toArray() {
        ArrayListI result = new ArrayListI();
        result.add(finals.length);

        for(int i = 0; i < finals.length; i++) {
            result.add(finals[i]);

            ArrayListI singles = new ArrayListI();
            ArrayListI rangeFrom = new ArrayListI();
            ArrayListI rangeTo = new ArrayListI();

            int[] keys = transitions[i].keysArray();
            sort(keys);
            for(int j = 0; j < keys.length; j++) {
                int from = j;
                int state = transitions[i].get(j);
                while(j + 1 < keys.length && keys[j + 1] == keys[j] + 1 && transitions[i].get(j + 1) == state)
                    j++;
                if(j == from) {
                    singles.add(keys[j]);
                } else {
                    rangeFrom.add(keys[from]);
                    rangeTo.add(keys[j]);
                }
            }

            result.add(rangeFrom.size());
            for(int j = 0; j < rangeFrom.size(); j++) {
                result.add(rangeFrom.get(j));
                result.add(rangeTo.get(j));
                result.add(transitions[i].get(rangeFrom.get(j)));
            }
            result.add(singles.size());
            for(int j = 0; j < singles.size(); j++) {
                result.add(singles.get(j));
                result.add(transitions[i].get(singles.get(j)));
            }
        }
        return result.toArray();
    }

    /* Used for debugging */
    @Override
    public String toString() {
        StringBuilder result = new StringBuilder();
        for(int i = 0; i < transitions.length; i++) {
            HashMapII map = transitions[i];

            IteratorI it = map.keys();
            while(it.hasNext()) {
                int v = it.next();
                result.append("edge(");
                result.append(i);
                result.append(", ");
                result.append(v);
                result.append("): ");
                result.append(map.get(v));
                result.append("\r\n");
            }
        }
        for(int i = 0; i < finals.length; i++) {
            if(finals[i] != DFA.NOFINAL) {
                result.append("final(");
                result.append(i);
                result.append("): ");
                result.append(finals[i]);
                result.append("\r\n");
            }
        }
        return result.toString();
    }
}
