package crafttweaker.api.data;

import java.util.*;

/**
 * Contains a list.
 *
 * @author Stan Hebben
 */
public class DataList implements IData {
    
    private final List<IData> values;
    private final boolean immutable;
    
    public DataList(List<IData> values, boolean immutable) {
        this.values = values;
        this.immutable = immutable;
    }
    
    @Override
    public IData getAt(int i) {
        return values.get(i);
    }
    
    @Override
    public void setAt(int i, IData value) {
        if(immutable) {
            throw new UnsupportedOperationException("this list is immutable");
        } else {
            values.set(i, value);
        }
    }
    
    @Override
    public IData memberGet(String name) {
        throw new UnsupportedOperationException("Lists don't have members");
    }
    
    @Override
    public void memberSet(String name, IData data) {
        throw new UnsupportedOperationException("Lists don't have members");
    }
    
    @Override
    public int length() {
        return values.size();
    }
    
    @Override
    public boolean contains(IData data) {
        List<IData> dataValues = data.asList();
        if(dataValues != null && containsList(dataValues))
            return true;
        
        for(IData value : values) {
            if(value.contains(data))
                return true;
        }
        
        return false;
    }
    
    private boolean containsList(List<IData> dataValues) {
        outer:
        for(IData dataValue : dataValues) {
            for(IData value : values) {
                if(value.contains(dataValue))
                    continue outer;
            }
            
            return false;
        }
        
        return true;
    }
    
    @Override
    public boolean equals(IData data) {
        List<IData> otherValues = data.asList();
        if (data.asList() == null) return false;
        if(otherValues.size() != values.size())
            return false;
        
        for(int i = 0; i < values.size(); i++) {
            if(!values.get(i).equals(otherValues.get(i)))
                return false;
        }
        
        return true;
    }
    
    @Override
    public int compareTo(IData data) {
        throw new UnsupportedOperationException("Cannot compare with a list");
    }
    
    @Override
    public IData immutable() {
        if(immutable) {
            return this;
        } else {
            List<IData> copy = new ArrayList<>();
            for(IData value : values) {
                copy.add(value.immutable());
            }
            return new DataList(copy, true);
        }
    }
    
    @Override
    public IData update(IData data) {
        if(immutable)
            data = data.immutable();
        
        List<IData> result = new ArrayList<>();
        result.addAll(values);
        result.addAll(data.asList());
        return new DataList(result, immutable);
    }
    
    @Override
    public <T> T convert(IDataConverter<T> converter) {
        return converter.fromList(values);
    }
    
    @Override
    public IData add(IData other) {
        return update(other);
    }
    
    @Override
    public IData sub(IData other) {
        throw new UnsupportedOperationException("Cannot subtract from a list");
    }
    
    @Override
    public IData mul(IData other) {
        throw new UnsupportedOperationException("Cannot multiply a list");
    }
    
    @Override
    public IData div(IData other) {
        throw new UnsupportedOperationException("Cannot divide a list");
    }
    
    @Override
    public IData mod(IData other) {
        throw new UnsupportedOperationException("Cannot perform modulo on a list");
    }
    
    @Override
    public IData and(IData other) {
        throw new UnsupportedOperationException("Lists don't support bitwise operators");
    }
    
    @Override
    public IData or(IData other) {
        throw new UnsupportedOperationException("Lists don't support bitwise operators");
    }
    
    @Override
    public IData xor(IData other) {
        throw new UnsupportedOperationException("Lists don't support bitwise operators");
    }
    
    @Override
    public IData neg() {
        throw new UnsupportedOperationException("Cannot negate a list");
    }
    
    @Override
    public IData not() {
        throw new UnsupportedOperationException("Lists don't support bitwise operators");
    }
    
    @Override
    public String toString() {
        return asString();
    }

    // binary compat
    @Override
    public boolean asBool() {
        return IData.super.asBool();
    }

    @Override
    public byte asByte() {
        return IData.super.asByte();
    }

    @Override
    public short asShort() {
        return IData.super.asShort();
    }

    @Override
    public int asInt() {
        return IData.super.asInt();
    }

    @Override
    public long asLong() {
        return IData.super.asLong();
    }

    @Override
    public float asFloat() {
        return IData.super.asFloat();
    }

    @Override
    public double asDouble() {
        return IData.super.asDouble();
    }

    @Override
    public String asString() {
        return IData.super.asString();
    }

    @Override
    public List<IData> asList() {
        return IData.super.asList();
    }

    @Override
    public Map<String, IData> asMap() {
        return IData.super.asMap();
    }

    @Override
    public byte[] asByteArray() {
        return IData.super.asByteArray();
    }

    @Override
    public int[] asIntArray() {
        return IData.super.asIntArray();
    }
}
